/************************************************************************************************************\

Module Name:    H264ParserModule.c

Description:    .

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                  I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "H264ParserModule.h"

// -----------------------------------------------------------------------------------------------------------
//                         N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                        S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------------------------------------

static const    MCHAR8      g_szModuleNameBase[]    = "H264Parser";
static          MUINT32     g_uiH264ParserModCount  = 0;

// -----------------------------------------------------------------------------------------------------------
//                                                  C O D E
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

Function:       H264ParserMod_Init

Description:    .

\************************************************************************************************************/
LStatus H264ParserMod_Init(
            H264ParserModule*   poH264ParserMod,
            LDevice_Handle      hDevice,
            MUINT               uiOutBufferCount,
            MUINT               uiMaxNaluSizeBytes,
            MUINT               uiFpsNumerator,
            MUINT               uiFpsDenominator)
{
    MsgLog(2, "{...");

    H264ParserMod_Cleanup(poH264ParserMod);

    LStatus eStatus = ((poH264ParserMod != MNULL)
                       && (hDevice != MNULL)
                       && (uiMaxNaluSizeBytes > 0)
                       && (uiFpsNumerator != 0)
                       && (uiFpsDenominator != 0))
                      ? LStatus_OK : LStatus_INVALID_PARAM;

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        LBuffer_Attributes oBufferAttribs;

        oBufferAttribs.oLinearAttributes.eAttributeType = LBuffer_Type_LINEAR;
        oBufferAttribs.oLinearAttributes.uiSize         = uiMaxNaluSizeBytes;

        snprintf(
                    poH264ParserMod->szModuleName,
                    sizeof(poH264ParserMod->szModuleName),
                    "%s%d",
                    g_szModuleNameBase,
                    g_uiH264ParserModCount);

        eStatus = ModLnk_Init(
                        &(poH264ParserMod->oOutLink),
                        hDevice,
                        uiOutBufferCount,
                        &oBufferAttribs,
                        MFALSE,
                        0,
                        poH264ParserMod->szModuleName);
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        poH264ParserMod->uiMaxNaluSizeBytes = uiMaxNaluSizeBytes;
        poH264ParserMod->uiFpsNumerator     = uiFpsNumerator;
        poH264ParserMod->uiFpsDenominator   = uiFpsDenominator;
    }
    else
    {
        H264ParserMod_Cleanup(poH264ParserMod);
    }

    ++g_uiH264ParserModCount;

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       H264ParserMod_Cleanup

Description:    .

\************************************************************************************************************/
void H264ParserMod_Cleanup(H264ParserModule* poH264ParserMod)
{
    MsgLog(2, "{...");

    if (poH264ParserMod != MNULL)
    {
        ModLnk_Cleanup(&(poH264ParserMod->oOutLink));

        poH264ParserMod->oInLink.poModLnk   = MNULL;
        poH264ParserMod->uiMaxNaluSizeBytes = 0;
    }

    MsgLog(2, "...}");
}

/************************************************************************************************************\

Function:       H264ParserMod_GetDstBuffer

Description:    .

\************************************************************************************************************/
static LStatus H264ParserMod_GetDstBuffer(
    H264ParserModule*   poH264ParserMod,
    BufferInfo**        ppoDstBuffer,
    MUINT8**            ppuiDst)
{
    LStatus eStatus = LStatus_FAIL;

    if ((poH264ParserMod != MNULL) && (ppoDstBuffer != MNULL))
    {
        while (!poH264ParserMod->oCpuThread.bKillThread)
        {
            eStatus = ModLnk_GetReturnedBuffer(
                        &(poH264ParserMod->oOutLink),
                        100,
                        poH264ParserMod->hDevThread,
                        ppoDstBuffer);

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                if (ppuiDst != MNULL)
                {
                    eStatus = LBuffer_BeginAccess((*ppoDstBuffer)->hBuffer, 0, 1, ppuiDst);
                }

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    break;
                }
                else
                {
                    MsgLogErr(
                        "LBuffer_BeginAccess failed! (status= %d - %s)",
                        eStatus,
                        GetStatusStr(eStatus));
                    ModLnk_ReleaseBuffer(&(poH264ParserMod->oOutLink), *ppoDstBuffer);
                }
            }

            if (eStatus != LStatus_TIMEOUT)
            {
                usleep(1000);
            }
        }
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       H264ParserMod_SubmitDstBuffer

Description:    .

\************************************************************************************************************/
static void H264ParserMod_SubmitDstBuffer(
    H264ParserModule*   poH264ParserMod,
    BufferInfo*         poDstBuffer,
    MBOOL               bEndAccess,
    MUINT               uiSizeBytes)
{
    if ((poH264ParserMod != MNULL) && (poDstBuffer != MNULL))
    {
        poDstBuffer->uiSizeBytes    = uiSizeBytes;
        poDstBuffer->uiStartOffset  = 0;

        if (bEndAccess)
        {
            LBuffer_EndAccess(poDstBuffer->hBuffer);
        }

        ModLnk_SubmitBuffer(&(poH264ParserMod->oOutLink), poDstBuffer, MNULL, NO_TAG);
    }
}

/************************************************************************************************************\

Function:       H264ParserMod_CpuThread

Description:    .

\************************************************************************************************************/
LStatus H264ParserMod_CpuThread(void* pvData)
{
    if (pvData == MNULL)
    {
        MsgLogErr("ERROR! NULL data.");
        return LStatus_INVALID_PARAM;
    }

    H264ParserModule*   poH264ParserMod     = (H264ParserModule*)pvData;
    BufferInfo*         poDstBuffer         = MNULL;
    MUINT8*             puiDst              = MNULL;
    MUINT               uiDstSize           = 0;
    MUINT               uiPreZeroCount      = 0;
    MUINT8              uiNaluType          = 0;
    MUINT64             uiNextTimestampUsec = 0;
    MUINT64             uiFrameDurationUsec = ((MUINT64)poH264ParserMod->uiFpsDenominator * 1000 * 1000)
                                              / poH264ParserMod->uiFpsNumerator;

    ModThread_SetName(poH264ParserMod->szModuleName);
    MsgLog(2, "Start thread %p.", pthread_self());

    while (!poH264ParserMod->oCpuThread.bKillThread)
    {
        BufferInfo* poSrcBuffer = MNULL;

        LStatus eStatus = ModLnkIn_GetSubmittedBuffer(
                                &(poH264ParserMod->oInLink),
                                100,
                                0,
                                poH264ParserMod->hDevThread,
                                &poSrcBuffer,
                                MNULL,
                                MNULL);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            if (poSrcBuffer->bEndOfStream)
            {
                if (poDstBuffer != MNULL)
                {
                    H264ParserMod_SubmitDstBuffer(poH264ParserMod, poDstBuffer, MTRUE, uiDstSize);

                    poDstBuffer = MNULL;
                    puiDst      = MNULL;
                    uiDstSize   = 0;
                }

                MsgLog(4, "END-OF-STREAM");

                eStatus = H264ParserMod_GetDstBuffer(poH264ParserMod, &poDstBuffer, MNULL);

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    poDstBuffer->bEndOfStream = MTRUE;
                }

                poH264ParserMod->oCpuThread.bKillThread = MTRUE;
            }
            else
            {
                MUINT8* puiSrc = MNULL;

                eStatus = LBuffer_BeginAccess(poSrcBuffer->hBuffer, 0, 1, &puiSrc);

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    puiSrc += poSrcBuffer->uiStartOffset;

                    MUINT  uiSrcSize        = poSrcBuffer->uiSizeBytes;
                    MUINT8 uiBeforeLastByte = (uiSrcSize > 1) ? puiSrc[uiSrcSize - 2] : ~0;
                    MUINT8 uiLastByte       = (uiSrcSize > 0) ? puiSrc[uiSrcSize - 1] : ~0;
                    MUINT  uiNextNaluOffset = GetNextH264StartCode(puiSrc, uiSrcSize, &uiPreZeroCount);

                    while (uiSrcSize > 0)
                    {
                        if (uiNextNaluOffset == 0)
                        {
                            // New NALU. So submit the previous one.
                            if (poDstBuffer != MNULL)
                            {
                                // If the nalu size is too short, ignore it and reuse the same destination
                                // buffer.
                                if (uiDstSize > (3 + uiPreZeroCount))
                                {
                                    poDstBuffer->uiTimestampUsec = uiNextTimestampUsec;

                                    // Synchronize video frames.
                                    if ((uiNaluType == 1) || (uiNaluType == 5))
                                    {
                                        uiNextTimestampUsec += uiFrameDurationUsec;
                                    }

                                    H264ParserMod_SubmitDstBuffer(
                                                poH264ParserMod,
                                                poDstBuffer,
                                                MTRUE,
                                                uiDstSize - uiPreZeroCount);

                                    poDstBuffer = MNULL;
                                    puiDst      = MNULL;
                                }

                                uiDstSize = 0;
                            }

                            // Find the next nalu offset.
                            MUINT uiNaluStart = 3 - uiPreZeroCount;

                            if (uiNaluStart < uiSrcSize)
                            {
                                uiNaluType       = puiSrc[uiNaluStart] & 0x1F;
                                uiNextNaluOffset = GetNextH264StartCode(
                                                        puiSrc + uiNaluStart,
                                                        uiSrcSize - uiNaluStart,
                                                        MNULL)
                                                   + uiNaluStart;
                            }
                            else
                            {
                                // The nalu type will be in the next source buffer.
                                uiNaluType       = ~0;
                                uiNextNaluOffset = uiSrcSize;
                            }
                        }
                        else if (uiNaluType == ~0)
                        {
                            // We stand here when the previous buffer end with a start code and so, this
                            // buffer start with the first byte of the nalu data.
                            uiNaluType = puiSrc[0] & 0x1F;
                        }

                        // Get a new destination buffer.
                        if (poDstBuffer == MNULL)
                        {
                            eStatus = H264ParserMod_GetDstBuffer(poH264ParserMod, &poDstBuffer, &puiDst);

                            if (LSTATUS_IS_FAIL(eStatus))
                            {
                                break;
                            }
                        }

                        if ((uiDstSize + uiPreZeroCount + uiNextNaluOffset)
                            > poH264ParserMod->uiMaxNaluSizeBytes)
                        {
                            MsgLogErr("ERROR! Nalu size larger than destination buffer. (%u > %u)",
                                      uiDstSize + uiPreZeroCount + uiNextNaluOffset,
                                      poH264ParserMod->uiMaxNaluSizeBytes);
                            eStatus = LStatus_FAIL;
                            break;
                        }

                        // Fill zero's from the end of the last buffer.
                        while (uiPreZeroCount > 0)
                        {
                            puiDst[uiDstSize] = 0;
                            uiDstSize++;
                            uiPreZeroCount--;
                        }

                        // Fill current nalu.
                        memcpy(puiDst + uiDstSize, puiSrc, uiNextNaluOffset);

                        uiDstSize       += uiNextNaluOffset;
                        uiSrcSize       -= uiNextNaluOffset;
                        puiSrc          += uiNextNaluOffset;
                        uiNextNaluOffset = 0;
                    }

                    uiPreZeroCount = (uiLastByte == 0) ? ((uiBeforeLastByte == 0) ? 2 : 1) : 0;

                    LBuffer_EndAccess(poSrcBuffer->hBuffer);
                }
                else
                {
                    MsgLogErr("LBuffer_BeginAccess(Buffer[%u]) FAILED! (status = %d, %s)",
                              poSrcBuffer->uiId, eStatus, GetStatusStr(eStatus));
                }
            }

            ModLnkIn_ReturnBuffer(&(poH264ParserMod->oInLink), poSrcBuffer, MNULL, NO_TAG);
        }

        if (LSTATUS_IS_FAIL(eStatus)
            && (eStatus != LStatus_TIMEOUT))
        {
            usleep(1000);
        }
    }

    if (poDstBuffer != MNULL)
    {
        H264ParserMod_SubmitDstBuffer(poH264ParserMod, poDstBuffer, (puiDst != MNULL), uiDstSize);
    }

    MsgLog(2, "Kill thread.");

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       H264ParserMod_Start

Description:    .

\************************************************************************************************************/
LStatus H264ParserMod_Start(H264ParserModule* poH264ParserMod)
{
    LStatus eStatus = LStatus_INVALID_PARAM;

    MsgLog(2, "{...");

    if ((poH264ParserMod != MNULL)
        && (poH264ParserMod->uiMaxNaluSizeBytes > 0))
    {
        if ((poH264ParserMod->oInLink.poModLnk != MNULL)
            && (poH264ParserMod->oOutLink.uiSubmitCount > 0))
        {
            eStatus = ModThread_Start(&(poH264ParserMod->oCpuThread), poH264ParserMod, H264ParserMod_CpuThread);
        }
        else
        {
            MsgLogErr("ERROR! Bad connection.");
            eStatus = LStatus_FAIL;
        }
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       H264ParserMod_Stop

Description:    .

\************************************************************************************************************/
void H264ParserMod_Stop(H264ParserModule* poH264ParserMod)
{
    MsgLog(2, "{...");

    if (poH264ParserMod != MNULL)
    {
        ModThread_Stop(&(poH264ParserMod->oCpuThread));
    }

    MsgLog(2, "...}");
}

